<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\OTP;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Mail\InformasiAkunMail;
use App\Mail\OTPVerificationMail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;

class OTPController extends Controller
{
    public function index($uuid)
    {
        $otp = OTP::where('uuid', $uuid)->firstOrFail();
        return view('password.otp', compact('otp'));
    }
    public function lupaAkun()
    {
        return view('password.lupa');
    }
    public function sendOTP(Request $request){
        $request->validate([
            'induk'     => 'required|string|exists:users,nomor_induk',
        ],[
            'induk.required'    => 'Nomor induk harus diisi!',
            'induk.exists'      => 'Nomor induk tidak terdaftar di sistem!',
        ]);
        $user = User::where('nomor_induk', $request->induk)->first();
        $otpAkun = OTP::where('id_user', $user->id)
                    ->where('verifikasi', false)
                    ->first();
        if ($otpAkun) {
            return back()->withErrors('OTP masih aktif, silahkan verifikasi terlebih dahulu!');
        }
        $otp = Str::random(6);
        $uuid = Str::uuid();  // Membuat UUID
        $otpRecord = OTP::Create(
            ['id_user' => $user->id,
                'kode_otp' => $otp,
                'uuid' => $uuid,
                'expired_at' => Carbon::now()->addMinutes(10), // OTP kadaluarsa dalam 10 menit
            ]
        );
        Mail::to($user->email)->send(new OTPVerificationMail($otp, $uuid));
        return redirect()->route('otp.index', ['uuid' => $uuid])->withSuccess('OTP telah dikirim ke email Anda');
    }

    public function verifyOTP(Request $request)
    {
        $this->validate($request, [
            'otp' => 'required|string',
            'uuid' => 'required|uuid',  // Verifikasi UUID
        ]);

        // Cari record OTP berdasarkan UUID dan email
        $otpRecord = OTP::where('uuid', $request->uuid)  // Cek UUID
                    ->first();

        if (!$otpRecord) {
            return back()->withErrors(['otp'    => 'OTP tidak ditemukan']);
        }

        if (Carbon::parse($otpRecord->expired_at)->lt(now())) {
            DB::transaction(function () use ($otpRecord) {
                $otpRecord->delete();
            });

            return back()->withErrors(['otp' => 'OTP telah kedaluarsa. Silahkan isi kembali formulir ganti password!']);
        }
        if ($otpRecord->verifikasi) {
            return back()->with('info', 'OTP sudah pernah diverifikasi sebelumnya.');
        }

        $otpRecord->update(['verifikasi' => true]);
        $user = User::where('id', $otpRecord->id_user)->firstOrFail();
        $password = Str::random(8);
        $user->update(['password' => bcrypt($password)]);
        Mail::to($user->email)->send(new InformasiAkunMail($user->nomor_induk, $password, $user->nama));
        return redirect()->route('login')->with('success','OTP terverifikasi, akun berhasil dikirim ke email');
    }
}
